package ga.core.algorithm.util;

import ga.core.GA;
import ga.core.individual.IClusterableIndividual;
import ga.core.individual.IFitness;
import ga.core.individual.IIndividual;
import ga.core.individual.IIntervalFitness;
import ga.core.validation.GAContext;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.math.stat.clustering.Cluster;

/**
 * Utility class for handling clusters.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class ClusterUtil {

  /**
   * Instantiation is not allowed.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private ClusterUtil() {
  }

  /**
   * Calculates the centroid of the collection. This is the individual of the
   * collection, that has the lowest distance to all other individuals.
   * 
   * @param c
   *          Collection of individuals.
   * @param <T>
   *          The generic type of the individuals.
   * @return The centroid individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static <T extends IClusterableIndividual<T>> T calculateCentroid(
      final Collection<T> c) {
    double minDist = Double.MAX_VALUE;
    T minP = null;

    // iterate through c
    final Iterator<T> it = c.iterator();
    while (it.hasNext()) {
      // test every point p1
      final T p1 = it.next();
      double totalDist = 0d;
      for (final T p2 : c) {
        // sum up the distance to all points p2 | p2!=p1
        if (p2 != p1) {
          totalDist += p1.distanceFrom(p2);
        }
      }

      // if the current distance is lower that the min, take it as new min
      if (totalDist < minDist) {
        minDist = totalDist;
        minP = p1;
      }
    }
    return minP;
  }

  /**
   * Assigns the fitness depending on the type of individual.
   * 
   * @param clusters
   *          The list of all clusters.
   * @param ind
   *          The individual that already has fitness assigned.
   * @param <T>
   *          The generic type of the individuals.
   * 
   * @see #assignNumericFitness(Cluster, IFitness)
   * @see #assignIntervalFitness(Cluster, IIntervalFitness)
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static <T extends IClusterableIndividual<T>> void assignFitness(
      final List<Cluster<T>> clusters, final T ind) {
    for (final Cluster<T> c : clusters) {
      if (ind.equals(c.getCenter())) {
        if (ind instanceof IIntervalFitness) {
          assignIntervalFitness(c, (IIntervalFitness) ind);
        } else {
          assignNumericFitness(c, ind);
        }

        return;
      }
    }

    throw new RuntimeException("something goes wrong");
  }

  /**
   * Assigns an interval fitness if the individual is an instance of
   * {@link IIntervalFitness}.
   * 
   * @param cluster
   *          The cluster.
   * @param ind
   *          The centroid of the cluster.
   * @param <T>
   *          The generic type of the individuals.
   * @param <I>
   *          The generic type of the interval fitness individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  @SuppressWarnings("rawtypes")
  public static <T extends IClusterableIndividual<T>, I extends IIntervalFitness> void assignIntervalFitness(
      final Cluster<T> cluster, final I ind) {
    final double f0 = ind.getFitness();

    for (final T i : cluster.getPoints()) {
      if (i instanceof IIntervalFitness) {

        float maxIntervalWidth = 10f;

        final GAContext context = ((IIndividual) ind).getContext();

        if (context != null) {
          maxIntervalWidth = context.getFloat(
              GA.KEY_INTERVAL_FITNESS_MAX_WIDTH, maxIntervalWidth);
        }

        @SuppressWarnings("unchecked")
        final double w = i.distanceFrom((T) ind);

        ((IIntervalFitness) i).setFitnessInterval(f0, maxIntervalWidth * w);
      }
    }

  }

  /**
   * Assigns the fitness of one individual to all individuals of the cluster.
   * This is the simpliest assignment method.
   * 
   * @param cluster
   *          The cluster.
   * @param ind
   *          The centroid of the cluster.
   * @param <T>
   *          The generic type of the individuals.
   * @param <I>
   *          The generic type of the interval fitness individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static <T extends IClusterableIndividual<T>, I extends IFitness> void assignNumericFitness(
      final Cluster<T> cluster, final I ind) {
    // simple assign the fitness to all individuals in the cluster
    for (final T i : cluster.getPoints()) {
      i.setFitness(ind.getFitness());
    }
  }
}
